<?php

declare(strict_types=1);

namespace Erlage\Photogram\Structures;

final class TableDependencyGraph
{
    /**
     * @var TableDependencyNode[]
     */
    public $nodes = array();

    /**
     * @var array
     */
    private $registry = array();

    /**
     * get a node for table name. if not exists, will create it 
     */
    public function getDependencyNode(string $tableName): TableDependencyNode
    {
        if ($this -> nodeExists($tableName))
        {
            return $this -> nodes[$this -> nodeIndex($tableName)];
        }

        $container = new TableDependencyNode($tableName, array());

        $this -> registerNode($tableName);

        $this -> nodes[] = $container;

        return $container;
    }

    /**
     * Tells whether there are nodes to pop
     */
    public function canPop(): bool
    {
        return \count($this -> nodes) > 0;
    }

    /**
     * Pops a TableDependencyNode node from graph and return.
     * before popping nodes, ensure a success call to canPop
     */
    public function popNode(): TableDependencyNode
    {
        /**
         * @var TableDependencyNode
         */
        $node = \array_pop($this -> nodes);

        $this -> unRegisterNode($node -> getTableName());

        return $node;
    }

    /**
     * clear all nodes and registry
     */
    public function clear(): void
    {
        $this -> nodes = array();
        $this -> registry = array();
    }

    /*
    |--------------------------------------------------------------------------
    | private helpers
    |--------------------------------------------------------------------------
    */

    private function nodeExists(string $tableName): bool
    {
        return isset($this -> registry[$tableName]);
    }

    private function nodeIndex(string $tableName): int
    {
        return $this -> registry[$tableName];
    }

    private function registerNode(string $tableName): void
    {
        $this -> registry[$tableName] = \count($this -> nodes);
    }

    private function unRegisterNode(string $tableName): void
    {
        unset($this -> registry[$tableName]);
    }
}
